/*****************************************************************************
 *   power modes.c:  main C entry file for NXP LPC17xx Family Microprocessors
 *
 *   Copyright(C) 2010, NXP Semiconductor
 *   All rights reserved.
 *
 *   History
 *   2010  ver 1.00    Prelimnary version, first Release
 *
******************************************************************************/
#include "lpc177x_8x.h"
#include "type.h"
#include "uart.h"
#include "pmu.h"

#include <stdio.h>

extern int getkey(void);
extern int sendchar(int);


extern volatile uint8_t UART0Buffer[BUFSIZE], UART1Buffer[BUFSIZE];
extern volatile uint32_t UART0Count, UART1Count; 
void GetDeviceSerial(unsigned long *pid, unsigned long *sn1, unsigned long *sn2, unsigned long *sn3, unsigned long *sn4);

void delay (void) 
{
  unsigned int i;
  for (i = 0; i < 0x800000; i++) 
  {}
}


void menu (void) 
{
/*Menu Screen on Terminal*/
#if 1
	unsigned long pid, sn1, sn2, sn3, sn4;

 printf ("\n\r");
 printf ("Welcome to the LPC178x/7x Low Power Modes test program \n\r");
 printf ("Default Set-up is Active Mode, Core Freq 120 MHz, PLL0 Enabled, GPIO Enabled, PCLK/4, GPIO+Uart0 Enabled\n\r");
 printf ("Note: After programming the part, you must disconnect the debugger and then cycle power to get correct current measurements.\n\r");

GetDeviceSerial(&pid, &sn1, &sn2, &sn3, &sn4);
printf("Device ID: 0x%08x S/N: 0x%08x %08x %08x %08x\r\n", pid, sn1, sn2, sn3, sn4);

 printf ("\n\r");
 
 printf ("1. Active Mode, Core Freq  12 MHz, PLL0 Disabled, PCLK/4, GPIO+Uart0 Enabled\n\r");
 printf ("2. Active Mode, Core Freq  48 MHz, PLL0 Enabled,  PCLK/4, GPIO+Uart0 Enabled\n\r");
 printf ("3. Active Mode, Core Freq  96 MHz, PLL0 Enabled,  PCLK/4, GPIO+Uart0 Enabled\n\r");
 printf ("4. Active Mode, Core Freq 120 MHz, PLL0 Enabled,  PCLK/4, GPIO+Uart0 Enabled\n\r");
 printf ("\n\r");
 printf ("s. Sleep Mode, PCLK/4, GPIO+Uart0 Enabled\n\r");
 printf ("S. Deep Sleep Mode\n\r");
 printf ("p. Power-down Mode\n\r");
 printf ("P. Deep Power-down Mode\n\r");
#if BENCHMARK==1
 printf ("\n\r");
 printf ("b. Set Benchmark Mode (disable UART) after next selection\n\r");
#endif
 printf ("\n\r");
#endif
}

void FlushUART(void)
{
	volatile int i;

	LPC_UART0->IER = IER_THRE | IER_RLS;			/* Disable RBR */
	UARTSend( 0, (uint8_t *)UART0Buffer, UART0Count );
	for(i=0;i<100000;i++);
	LPC_UART0->IER = IER_THRE | IER_RLS | IER_RBR;	/* Re-enable RBR */
	for(i=0;i<10000;i++);
}

#if BENCHMARK==1
int PowerBenchmark;
#endif

void SetBenchmarkMode(void)
{
#if BENCHMARK==1
	if(!PowerBenchmark)
	{
		printf("Benchmark mode will be active after next mode selection.\r\n");
		PowerBenchmark=1;
	}
	else
	{
		printf("Benchmark mode disabled.\r\n");
		PowerBenchmark=0;
	}
#endif
}

int BenchDump(void)
{
#if BENCHMARK==1
	if(PowerBenchmark)
	{
		LPC_SC->PCONP = 0x00000000; // Turn off all power incl. UART
		LPC_SC->PCON |= 1<<4; // Turn off brown-out reset
		LPC_SC->PCON |= 1<<3; // Turn off brown-out always
		LPC_SC->PCON |= 1<<2; // Turn off brown-out in low-power modes

		{
			unsigned long OldPCON = LPC_SC->PCON;
			unsigned long OldPCONP = LPC_SC->PCONP;
			unsigned long OldSCR = SCB->SCR;
			unsigned long OldPLL0CON = LPC_SC->PLL0CON;
			unsigned long OldPLL0STAT = LPC_SC->PLL0FEED;
			unsigned long OldPCLKSEL = LPC_SC->PCLKSEL;
			unsigned long OldCCLKSEL = LPC_SC->CCLKSEL;
			unsigned long OldCLKSRCSEL = LPC_SC->CLKSRCSEL;
		
			LPC_SC->PCONP = 0x00008008;
			printf("Dump0: PCON=0x%08x PCONP=0x%08x SCR=0x%08x\r\n",
				OldPCON, OldPCONP, OldSCR);
			printf("Dump1: PLL0CON=0x%08x PLL0STAT=0x%08x PCLKSEL=0x%08x CCLKSEL=0x%08x CLKSRCSEL=0x%08x\r\n",
				OldPLL0CON, OldPLL0STAT, OldPCLKSEL, OldCCLKSEL, OldCLKSRCSEL);
			printf("Benchmark mode active. Press reset to exit test.\r\nMeasure me -->");
			FlushUART();
		
			LPC_SC->PCONP = OldPCONP;
			return 1;
		}
	}
	return 0;
#endif
}

/*****************************************************************************
**   Main Function  main()
******************************************************************************/
int main (void)
{

  int i;
  
#if 1
	__IO uint32_t *p;

/*configure pins as GPIOs*/
for(p=&LPC_IOCON->P0_0;p<=&LPC_IOCON->P5_4;p++)
{
	*p &= 0x1F; // Select pin function 0 (GPIO) and no PU/PD
}

/*configure pins as GPIOs outputs*/
   LPC_GPIO0->DIR	= 0xFFFFFFFF;
   LPC_GPIO1->DIR	= 0xFFFFFFFF;
   LPC_GPIO2->DIR	= 0xFFFFFFFF;
   LPC_GPIO3->DIR	= 0xFFFFFFFF;
   LPC_GPIO4->DIR	= 0xFFFFFFFF;
   LPC_GPIO5->DIR	= 0xFFFFFFFF;

/*Drive all GPIOs low*/
   LPC_GPIO0->CLR	= 0xFFFFFFFF;
   LPC_GPIO1->CLR	= 0xFFFFFFFF;
   LPC_GPIO2->CLR	= 0xFFFFFFFF;
   LPC_GPIO3->CLR	= 0xFFFFFFFF;
   LPC_GPIO4->CLR	= 0xFFFFFFFF;
   LPC_GPIO5->CLR	= 0xFFFFFFFF;
#endif


#if 0
/* P1.27 -> CLKOUT */
LPC_IOCON->PINSEL3 |= 0x00400000;
//CCLK clkout
LPC_SC->CLKOUTCFG = 0x100;
#endif

	 LPC_SC->USBCLKSEL = 0;
	 LPC_SC->EMCCLKSEL = 0;

/*Set all peripherals clocks to pcclk = cclk/4 */
LPC_SC->PCLKSEL = 4;


/*Initialize UART0 interface at baud-rate 9600*/
#if 1
  UARTInit(0, 9600);	/* baud rate setting */
#endif

/*Turn off all peripherals except UART0 + GPIO*/
LPC_SC->PCONP = 0x00008008;

/*Disable Brownout detect during powerdown modes */
LPC_SC->PCON     |= 0x4;

PMU_Init();

menu();

for(i=0;i<20000;i++);	/*delay*/


/*Default Set-up is Active Mode, 120 MHz, PLL Enabled, Uart Running*/	  
LPC_SC->PCONP = 0x00000008; // disable GPIO registers

while (1) /* Loop forever */
{					
#if 1
  if (UART0Count != 0)
   {
     LPC_UART0->IER = IER_THRE | IER_RLS;			/* Disable RBR */
     UARTSend(0, (uint8_t *)UART0Buffer, UART0Count);
     UART0Count = 0;
     LPC_UART0->IER = 0x0;

#if BENCHMARK==1
  if (UART0Buffer[0] == 'b')	
   	{
		SetBenchmarkMode();
   		FlushUART();
	}
#endif	                    

/*Active Mode, 12 MHz, PLL0 Disabled, Uart0 Running*/	  	 
  if (UART0Buffer[0] == '1')	
   	{
   		FlushUART();
	                    
	 //Disconnect and turn off the PLL	 	
	 LPC_SC->CCLKSEL = 1;
	 LPC_SC->PLL0CON     &= ~(1<<0);                          /*  Disable the main PLL (PLL0) */
	 LPC_SC->PLL0FEED    = 0xAA;                               /*  Feed */
	 LPC_SC->PLL0FEED    = 0x55;                                 /* Feed */
	 while ((LPC_SC->PLL0STAT & (1<<8)) != 0x00);       /*  Wait for main PLL (PLL0) to shut down */

	 SystemCoreClockUpdate();
	 UARTInit(0, 9600);	/* baud rate setting */
	 UART0Count = 0;
	 LPC_SC->FLASHCFG  = 0x00000000 | 0x03A;
 	 printf("Active mode at %d MHz, PCLK/%d\r\n", SystemCoreClock / 1000000, LPC_SC->PCLKSEL&0x1F);
	 if(BenchDump())
	 	while(1);
   }

/*Active Mode, 48 MHz, PLL0 enabled, Uart0 Running*/	  	 
  if (UART0Buffer[0] == '2')	
   {
   		FlushUART();
	                    
	 LPC_SC->FLASHCFG  = 0x00005000 | 0x03A;
	 //Disconnect and turn off the PLL0	 	
	 LPC_SC->CCLKSEL = 1;
	 LPC_SC->PLL0CON     &= ~(1<<0);                          /*  Disable the main PLL (PLL0) */
	 LPC_SC->PLL0FEED    = 0xAA;                               /*  Feed */
	 LPC_SC->PLL0FEED    = 0x55;                                 /* Feed */
	 while ((LPC_SC->PLL0STAT & (1<<8)) != 0x00);       /*  Wait for main PLL (PLL0) to shut down */

	 //Connect and Turn off the PLL0	to set-up for 48 MHz
	 LPC_SC->CLKSRCSEL = 1;    /* Select Clock Source for PLL0       */
  	 LPC_SC->PLL0CFG   = 0x00000023;
     LPC_SC->PLL0CON  |= 0x01;             /* PLL0 Enable                        */
     LPC_SC->PLL0FEED  = 0xAA;
     LPC_SC->PLL0FEED  = 0x55;
     while (!(LPC_SC->PLL0STAT & (1<<10)));/* Wait for PLOCK0                    */
	 LPC_SC->CCLKSEL = 1 | (1<<8);

	 SystemCoreClockUpdate();
	 UARTInit(0, 9600);	/* baud rate setting */
	 UART0Count = 0;
	 LPC_SC->FLASHCFG  = 0x00002000 | 0x03A;
 	 printf("Active mode at %d MHz, PCLK/%d\r\n", SystemCoreClock / 1000000, LPC_SC->PCLKSEL&0x1F);
	 if(BenchDump())
	 	while(1);
   }

/*Active Mode, 96 MHz, PLL0 enabled, Uart0 Running*/	  	 
  if (UART0Buffer[0] == '3')	
   {
   		FlushUART();
	                    
	 LPC_SC->FLASHCFG  = 0x00005000 | 0x03A;
	 //Disconnect and turn off the PLL0	 	
	 LPC_SC->CCLKSEL = 1;
	 LPC_SC->PLL0CON     &= ~(1<<0);                          /*  Disable the main PLL (PLL0) */
	 LPC_SC->PLL0FEED    = 0xAA;                               /*  Feed */
	 LPC_SC->PLL0FEED    = 0x55;                                 /* Feed */
	 while ((LPC_SC->PLL0STAT & (1<<8)) != 0x00);       /*  Wait for main PLL (PLL0) to shut down */

	 //Connect and Turn off the PLL0	to set-up for 96 MHz
	 LPC_SC->CLKSRCSEL = 1;    /* Select Clock Source for PLL0       */
  	 LPC_SC->PLL0CFG   = 0x00000007;
     LPC_SC->PLL0CON   |= 0x01;             /* PLL0 Enable                        */
     LPC_SC->PLL0FEED  = 0xAA;
     LPC_SC->PLL0FEED  = 0x55;
     while (!(LPC_SC->PLL0STAT & (1<<10)));/* Wait for PLOCK0                    */
	 LPC_SC->CCLKSEL = 1 | (1<<8);

	 SystemCoreClockUpdate();
	 UARTInit(0, 9600);	/* baud rate setting */
	 UART0Count = 0;
	 LPC_SC->FLASHCFG  = 0x00004000 | 0x03A;
 	 printf("Active mode at %d MHz, PCLK/%d\r\n", SystemCoreClock / 1000000, LPC_SC->PCLKSEL&0x1F);
	 if(BenchDump())
	 	while(1);
   }

/*Active Mode, 120 MHz, PLL0 enabled, Uart0 Running*/	  	 
  if (UART0Buffer[0] == '4')	
   {
   		FlushUART();
	                    
	 LPC_SC->FLASHCFG  = 0x00005000 | 0x03A;
	 //Disconnect and turn off the PLL0	 	
	 LPC_SC->CCLKSEL = 1;
	 LPC_SC->PLL0CON     &= ~(1<<0);                          /*  Disable the main PLL (PLL0) */
	 LPC_SC->PLL0FEED    = 0xAA;                               /*  Feed */
	 LPC_SC->PLL0FEED    = 0x55;                                 /* Feed */
	 while ((LPC_SC->PLL0STAT & (1<<8)) != 0x00);       /*  Wait for main PLL (PLL0) to shut down */

	 //Connect and Turn off the PLL0	to set-up for 96 MHz
	 LPC_SC->CLKSRCSEL = 1;    /* Select Clock Source for PLL0       */
  	 LPC_SC->PLL0CFG   = 0x00000009;
     LPC_SC->PLL0CON   = 0x01;             /* PLL0 Enable                        */
     LPC_SC->PLL0FEED  = 0xAA;
     LPC_SC->PLL0FEED  = 0x55;
     while (!(LPC_SC->PLL0STAT & (1<<10)));/* Wait for PLOCK0                    */
	 LPC_SC->CCLKSEL = 1 | (1<<8);

	 SystemCoreClockUpdate();
	 UARTInit(0, 9600);	/* baud rate setting */
	 LPC_SC->FLASHCFG  = 0x00005000 | 0x03A;
 	 printf("Active mode at %d MHz, PCLK/%d\r\n", SystemCoreClock / 1000000, LPC_SC->PCLKSEL&0x1F);
	 if(BenchDump())
	 	while(1);
   }

	  
/*Sleep Mode*/	  	 
	if (UART0Buffer[0] == 's')
	{
   		printf("Entering sleep at %d MHz, PCLK/%d\r\n", SystemCoreClock / 1000000, LPC_SC->PCLKSEL&0x1F);
		printf("Press any key, pull P2[10] low, or reset to exit.\r\n");
   		FlushUART();

		if(PMU_Sleep(MCU_SLEEP))
			printf("Exited sleep to active mode at %d MHz, PCLK/%d\r\n", SystemCoreClock / 1000000, LPC_SC->PCLKSEL&0x1F);
		else
			printf("Failed to enter sleep mode\r\n");
		UART0Count = 0;
   } else if (UART0Buffer[0] == '?')	
		menu();


/*Deep-Sleep Mode*/ 
  else if (UART0Buffer[0] == 'S')
   {
   		printf("Entering deep sleep at %d MHz\r\n", SystemCoreClock / 1000000);
		printf("Pull P2[10] low, or reset to exit.\r\n");
		FlushUART();

		if(PMU_Sleep(MCU_DEEP_SLEEP))
		{
			SystemCoreClockUpdate();
			UARTInit(0, 9600);	/* baud rate setting */
	   		printf("Exited deep sleep to active mode at %d MHz, PCLK/%d\r\n", SystemCoreClock / 1000000, LPC_SC->PCLKSEL&0x1F);
		} else
			printf("Failed to enter deep sleep mode\r\n");
		UART0Count = 0;
	}

/*Power-down mode*/ 
 else if (UART0Buffer[0] == 'p')
	  {
   		printf("Entering power-down mode\r\n");
		printf("Reset to exit. Pulling P2[10] low exits to ISP mode.\r\n");
		FlushUART();

//		MinimumSystemInit();
		if(PMU_PowerDown(MCU_POWERDOWN))
		{
			SystemCoreClockUpdate();
			UARTInit(0, 9600);	/* baud rate setting */
	   		printf("Exited power-down mode to active mode at %d MHz, PCLK/%d\r\n", SystemCoreClock / 1000000, LPC_SC->PCLKSEL&0x1F);
		} else
			printf("Failed to enter power-down mode.\r\n");

		UART0Count = 0;
	  }

/*Deep Power-down Mode*/
else if (UART0Buffer[0] == 'P')
	  {
   		printf("Entering deep power-down mode\r\n");
		printf("Reset to exit.\r\n");
		FlushUART();

//		MinimumSystemInit();
		if(PMU_PowerDown(MCU_DEEP_POWERDOWN))
			while(1); // code is unreachable; only exit from DPD is reset
		else
			printf("Failed to enter deep power-down mode.\r\n");
	  }
  
LPC_UART0->IER = IER_THRE | IER_RLS | IER_RBR;	// Re-enable RBR
	 
    }
#endif
  }

}

typedef void (*IAP)(unsigned long [],unsigned long[]);

void GetDeviceSerial(unsigned long *pid, unsigned long *sn1, unsigned long *sn2, unsigned long *sn3, unsigned long *sn4)
{
	IAP iap_entry = (IAP)0x1FFF1FF1;
	unsigned long command[5], result[5];

	command[0] = 54;
	result[0]=result[1]=result[2]=result[3]=result[4]=0;
	iap_entry(command, result);
	if(pid) *pid = result[1];

	command[0] = 58;
	result[0]=result[1]=result[2]=result[3]=result[4]=0;
	iap_entry(command, result);
	if(sn1) *sn1 = result[1];
	if(sn2) *sn2 = result[2];
	if(sn3) *sn3 = result[3];
	if(sn4) *sn4 = result[4];
}

/******************************************************************************
**                            End Of File
******************************************************************************/
